/*
 * Decompiled with CFR 0.152.
 */
package io.antmedia.enterprise.adaptive;

import io.antmedia.enterprise.adaptive.base.AudioEncoder;
import io.antmedia.enterprise.adaptive.base.Encoder;
import io.antmedia.enterprise.adaptive.base.VideoEncoder;
import io.antmedia.plugin.FrameFeeder;
import io.antmedia.plugin.api.IFrameListener;
import io.vertx.core.Vertx;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.bytedeco.ffmpeg.avcodec.AVPacket;
import org.bytedeco.ffmpeg.avutil.AVFrame;
import org.bytedeco.ffmpeg.global.avcodec;
import org.bytedeco.ffmpeg.global.avutil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StreamAdaptor {
    private static final String QUEUE_SIZE_IS_EXCEEDING_SO_DROPPING_FRAME = "Queue size({}) is exceeding {} so dropping frame for stream: {}";
    private static final String STREAM_ADAPTOR_QUEUE_SIZE = "Stream adaptor queue size: {} for stream: {}";
    private static final int MAX_QUEUE_SIZE = 2000;
    private static final Logger logger = LoggerFactory.getLogger(StreamAdaptor.class);
    private static final long serialVersionUID = 1L;
    ConcurrentLinkedQueue<Frame> frameQueue = new ConcurrentLinkedQueue();
    protected AtomicInteger queueSize = new AtomicInteger(0);
    List<VideoEncoder> videoEncoderList = new ArrayList<VideoEncoder>();
    List<AudioEncoder> audioEncoderList = new ArrayList<AudioEncoder>();
    protected AtomicBoolean isJobRunning = new AtomicBoolean(false);
    private long scheduledJobName;
    private volatile boolean stopRequest = false;
    private int videoIndex;
    private int audioIndex;
    private boolean recording;
    ConcurrentLinkedQueue<AVFrame> availableFrames = new ConcurrentLinkedQueue();
    ConcurrentLinkedQueue<AVPacket> availablePackets = new ConcurrentLinkedQueue();
    int logCounter = 0;
    private boolean streamable;
    private long totalVideoEncodeTime;
    private long videoPacketCount;
    private long totalAudioEncodeTime;
    private long audioPacketCount;
    private long numberOfThreadEntrance;
    private long lastEnteredTime;
    private long totalThreadEnterInterval;
    private Vertx vertx;
    private String streamId;
    private FrameFeeder frameFeeder;

    public StreamAdaptor(boolean streamable, Vertx vertx, String streamId) {
        this.streamable = streamable;
        this.vertx = vertx;
        this.streamId = streamId;
        this.frameFeeder = new FrameFeeder(streamId);
    }

    public void addVideoEncoder(VideoEncoder videoEncoder) {
        this.videoEncoderList.add(videoEncoder);
    }

    public void addAudioEncoder(AudioEncoder audioEncoder) {
        this.audioEncoderList.add(audioEncoder);
    }

    public List<VideoEncoder> getVideoEncoderList() {
        return this.videoEncoderList;
    }

    public List<AudioEncoder> getAudioEncoderList() {
        return this.audioEncoderList;
    }

    public void addQueue(AVFrame frame, int streamIndex, PacketType streamType, long captureTimeMS) {
        ++this.logCounter;
        if (this.logCounter % 1000 == 0) {
            logger.info(STREAM_ADAPTOR_QUEUE_SIZE, (Object)this.queueSize.get(), (Object)this.streamId);
            this.logCounter = 0;
        }
        if (this.queueSize.getAndIncrement() > 2000) {
            logger.info(QUEUE_SIZE_IS_EXCEEDING_SO_DROPPING_FRAME, new Object[]{this.queueSize.decrementAndGet(), 2000, this.streamId});
            return;
        }
        AVFrame dstFrame = this.getAVFrame();
        avutil.av_frame_ref((AVFrame)dstFrame, (AVFrame)frame);
        this.frameQueue.offer(new Frame(streamType, dstFrame, streamIndex, captureTimeMS));
    }

    public void addPacket2Queue(AVPacket avpacket, int streamIndex, PacketType streamType, long captureTimeMs) {
        ++this.logCounter;
        if (this.logCounter % 1000 == 0) {
            logger.info(STREAM_ADAPTOR_QUEUE_SIZE, (Object)this.queueSize.get(), (Object)this.streamId);
            this.logCounter = 0;
        }
        if (this.queueSize.getAndIncrement() > 2000) {
            logger.info(QUEUE_SIZE_IS_EXCEEDING_SO_DROPPING_FRAME, new Object[]{this.queueSize.decrementAndGet(), 2000, this.streamId});
            return;
        }
        AVPacket pkt = this.getAVPacket();
        avcodec.av_packet_ref((AVPacket)pkt, (AVPacket)avpacket);
        this.frameQueue.offer(new Frame(streamType, pkt, streamIndex, captureTimeMs));
    }

    public void start() {
        this.scheduledJobName = this.vertx.setPeriodic(10L, id -> this.vertx.executeBlocking(future -> {
            try {
                this.execute();
            }
            catch (Exception e) {
                logger.error(ExceptionUtils.getStackTrace((Throwable)e));
            }
            future.complete();
        }, false, null));
        logger.info("Stream adaptor scheduled job id {} stream adaptor:{} for stream:{}", new Object[]{this.scheduledJobName, this.hashCode(), this.streamId});
        this.recording = true;
    }

    public void stop() {
        this.stopRequest = true;
    }

    public void execute() {
        ++this.numberOfThreadEntrance;
        long now = System.currentTimeMillis();
        this.totalThreadEnterInterval += this.lastEnteredTime != 0L ? now - this.lastEnteredTime : 0L;
        this.lastEnteredTime = now;
        if (this.isJobRunning.compareAndSet(false, true)) {
            long refTime;
            while (!this.frameQueue.isEmpty()) {
                Frame frame = this.frameQueue.poll();
                this.queueSize.decrementAndGet();
                if (frame == null) continue;
                try {
                    if (frame.type == PacketType.VIDEO_FRAME) {
                        this.videoIndex = frame.streamIndex;
                        refTime = System.currentTimeMillis();
                        this.write2VideoEncoders(frame);
                        ++this.videoPacketCount;
                        this.totalVideoEncodeTime += System.currentTimeMillis() - refTime;
                        this.addFrame2Pool(frame);
                        continue;
                    }
                    if (frame.type == PacketType.AUDIO_FRAME) {
                        this.audioIndex = frame.streamIndex;
                        refTime = System.currentTimeMillis();
                        this.write2AudioEncoders(frame);
                        ++this.audioPacketCount;
                        this.totalAudioEncodeTime += System.currentTimeMillis() - refTime;
                        this.addFrame2Pool(frame);
                        continue;
                    }
                    if (frame.type != PacketType.VIDEO_PACKET) continue;
                    this.videoIndex = frame.streamIndex;
                    this.write2VideoEncoderBuffer(frame);
                    this.addPacket2Pool(frame);
                }
                catch (Exception e) {
                    logger.error(ExceptionUtils.getStackTrace((Throwable)e));
                }
            }
            if (this.stopRequest && this.recording) {
                logger.info("Recording set to false in streamadaptor: {} for stream: {} stopping scheduler job id: {}", new Object[]{this.hashCode(), this.streamId, this.scheduledJobName});
                this.vertx.cancelTimer(this.scheduledJobName);
                this.scheduledJobName = -1L;
                refTime = System.currentTimeMillis();
                this.flushVideoEncoders();
                this.totalVideoEncodeTime += System.currentTimeMillis() - refTime;
                refTime = System.currentTimeMillis();
                this.flushAudioEncoders();
                this.totalAudioEncodeTime += System.currentTimeMillis() - refTime;
                this.writeEncodeTrailers();
                this.releaseFrames();
                this.releasePackets();
                this.recording = false;
                this.logStats();
                logger.info("Leaving streamadaptor: {} for stream: {}", (Object)this.hashCode(), (Object)this.streamId);
            }
            this.isJobRunning.compareAndSet(true, false);
        }
    }

    private void write2VideoEncoderBuffer(Frame frame) {
        for (VideoEncoder videoEncoder : this.videoEncoderList) {
            frame.packet.position(0L);
            byte[] data = new byte[frame.packet.size()];
            frame.packet.data().get(data);
            ByteBuffer byteBuffer = ByteBuffer.allocateDirect(data.length);
            byteBuffer.put(data);
            int keyFrame = frame.packet.flags() & 1;
            videoEncoder.writeVideoBuffer(byteBuffer, frame.packet.dts(), 0, frame.streamIndex, keyFrame == 1, 0L, frame.packet.pts());
        }
    }

    public void write2AudioEncoders(Frame frame) throws Exception {
        AVFrame processedFrame = this.frameFeeder.onAudioFrame(frame.avFrame);
        if (processedFrame != null) {
            if (frame.avFrame != processedFrame) {
                avutil.av_frame_ref((AVFrame)frame.avFrame, (AVFrame)processedFrame);
            }
            for (AudioEncoder audioEncoder : this.audioEncoderList) {
                audioEncoder.writeFrame(frame.avFrame, frame.streamIndex, frame.captureTimeMS);
            }
        }
    }

    public void write2VideoEncoders(Frame frame) throws Exception {
        AVFrame processedFrame = this.frameFeeder.onVideoFrame(frame.avFrame);
        if (processedFrame != null) {
            if (frame.avFrame != processedFrame) {
                avutil.av_frame_ref((AVFrame)frame.avFrame, (AVFrame)processedFrame);
            }
            for (VideoEncoder videoEncoder : this.videoEncoderList) {
                videoEncoder.writeFrame(frame.avFrame, frame.streamIndex, frame.captureTimeMS);
            }
        }
    }

    private void addPacket2Pool(Frame frame) {
        avcodec.av_packet_unref((AVPacket)frame.packet);
        this.availablePackets.offer(frame.packet);
    }

    public void addFrame2Pool(Frame frame) {
        avutil.av_frame_unref((AVFrame)frame.avFrame);
        this.availableFrames.offer(frame.avFrame);
    }

    public void flushAudioEncoders() {
        for (AudioEncoder encoder : this.audioEncoderList) {
            try {
                while (encoder.writeFrame(null, this.audioIndex, System.currentTimeMillis())) {
                }
                logger.info("Flushing audio encoder:{} for stream adaptor:{} stream id:{}", new Object[]{encoder.getCodecName(), this.hashCode(), this.streamId});
            }
            catch (Exception e) {
                logger.error(ExceptionUtils.getStackTrace((Throwable)e));
            }
        }
    }

    public void flushVideoEncoders() {
        for (VideoEncoder encoder : this.videoEncoderList) {
            try {
                while (encoder.writeFrame(null, this.videoIndex, System.currentTimeMillis())) {
                }
                logger.info("Flushing video encoder:{} for stream adaptor:{} stream id:{}", new Object[]{encoder.getCodecName(), this.hashCode(), this.streamId});
            }
            catch (Exception e) {
                logger.error(ExceptionUtils.getStackTrace((Throwable)e));
            }
        }
    }

    public void logStats() {
        if (this.videoPacketCount > 0L && this.audioPacketCount > 0L) {
            long avarageVideoEncodeTime = this.totalVideoEncodeTime / this.videoPacketCount;
            long avarageAudioEncodeeTime = this.totalAudioEncodeTime / this.audioPacketCount;
            logger.info("Average video encode time {}ms video packet count: {} Average audio encode time {}ms audio packet count: {} for stream: {}", new Object[]{avarageVideoEncodeTime, this.videoPacketCount, avarageAudioEncodeeTime, this.audioPacketCount, this.streamId});
        }
        if (this.numberOfThreadEntrance > 0L) {
            long avarageThreadEntranceIntervalTime = this.totalThreadEnterInterval / this.numberOfThreadEntrance;
            logger.info("Average thread entrance interval {}ms for stream: {}", (Object)avarageThreadEntranceIntervalTime, (Object)this.streamId);
        }
    }

    public void releaseFrames() {
        for (AVFrame avFrame : this.availableFrames) {
            avutil.av_frame_unref((AVFrame)avFrame);
            avFrame.close();
        }
        this.availableFrames = null;
    }

    private void releasePackets() {
        for (AVPacket pkt : this.availablePackets) {
            avcodec.av_packet_unref((AVPacket)pkt);
            pkt.close();
        }
        this.availablePackets = null;
    }

    public void writeEncodeTrailers() {
        this.frameFeeder.writeTrailer();
        for (Encoder encoder : this.videoEncoderList) {
            encoder.writeTrailer();
        }
        for (Encoder encoder : this.audioEncoderList) {
            encoder.writeTrailer();
        }
    }

    public boolean isRecording() {
        return this.recording;
    }

    public AVFrame getAVFrame() {
        if (!this.availableFrames.isEmpty()) {
            return this.availableFrames.poll();
        }
        return new AVFrame();
    }

    private AVPacket getAVPacket() {
        if (!this.availablePackets.isEmpty()) {
            return this.availablePackets.poll();
        }
        return new AVPacket();
    }

    public boolean isStreamable() {
        return this.streamable;
    }

    public void setStreamable(boolean streamable) {
        this.streamable = streamable;
    }

    public void addFrameListener(IFrameListener listener) {
        this.frameFeeder.addListener(listener);
    }

    public void removeFrameListener(IFrameListener listener) {
        this.frameFeeder.removeFrameListener(listener);
    }

    public FrameFeeder getFrameFeeder() {
        return this.frameFeeder;
    }

    public static class Frame {
        PacketType type;
        AVFrame avFrame;
        AVPacket packet;
        int streamIndex;
        long captureTimeMS;

        public Frame(PacketType type, AVFrame frame, int streamIndex, long captureTimeMS) {
            this.type = type;
            this.avFrame = frame;
            this.streamIndex = streamIndex;
            this.captureTimeMS = captureTimeMS;
        }

        public Frame(PacketType type, AVPacket packet, int streamIndex, long captureTimeMS) {
            this.type = type;
            this.packet = packet;
            this.streamIndex = streamIndex;
            this.captureTimeMS = captureTimeMS;
        }
    }

    public static enum PacketType {
        VIDEO_FRAME,
        AUDIO_FRAME,
        VIDEO_PACKET;

    }
}

