document.addEventListener('DOMContentLoaded', () => {
    const btnExport = document.getElementById('btnExport');
    const btnImport = document.getElementById('btnImport');
    const fileImport = document.getElementById('fileImport');
    const statusMsg = document.getElementById('statusMsg');
    const folderCountEl = document.getElementById('folderCount');
    const linkCountEl = document.getElementById('linkCount');

    // Initial load
    loadStats();

    // Data Loading & Migration Logic
    function loadStats() {
        showStatus('🔄 檢查同步資料...', 'info', true);

        chrome.storage.sync.get(['folders', 'linkMappings', 'linkNotes'], (syncData) => {
            const syncFolders = syncData.folders || [];
            const syncMappings = syncData.linkMappings || {};
            const syncNotes = syncData.linkNotes || {};

            const syncHasData = syncFolders.length > 0 || Object.keys(syncMappings).length > 0 || Object.keys(syncNotes).length > 0;

            if (syncHasData) {
                updateUI(syncFolders, syncMappings);
                showStatus('✅ 資料已同步 (雲端)', 'success', true);
            } else {
                // Sync is empty, check Local
                showStatus('☁️ 雲端無資料，檢查本機...', 'info', true);

                chrome.storage.local.get(['folders', 'linkMappings', 'linkNotes'], (localData) => {
                    const localFolders = localData.folders || [];
                    const localMappings = localData.linkMappings || {};
                    const localNotes = localData.linkNotes || {};

                    const localHasData = localFolders.length > 0 || Object.keys(localMappings).length > 0;

                    if (localHasData) {
                        showStatus(`🔄 發現本機資料 (${localFolders.length})，上傳中...`, 'info', true);
                        console.log('Migrating local data to sync...');

                        // Attempt Migration
                        chrome.storage.sync.set({
                            folders: localFolders,
                            linkMappings: localMappings,
                            linkNotes: localNotes
                        }, () => {
                            if (chrome.runtime.lastError) {
                                console.error('Sync Error:', chrome.runtime.lastError);
                                if (chrome.runtime.lastError.message && chrome.runtime.lastError.message.includes('QUOTA')) {
                                    showStatus(`❌ 同步失敗：資料過多`, 'error', true);
                                } else {
                                    showStatus(`❌ 同步失敗：${chrome.runtime.lastError.message}`, 'error', true);
                                }
                                updateUI(localFolders, localMappings);
                            } else {
                                updateUI(localFolders, localMappings);
                                showStatus('✅ 資料已同步 (雲端)', 'success', true);
                            }
                        });
                    } else {
                        // Both empty
                        updateUI([], {});
                        showStatus('☁️ 目前無資料', 'info', true);
                    }
                });
            }
        });
    }

    function updateUI(folders, mappings) {
        // Filter out system folders for count display?
        // Usually we have 'all' and 'uncategorized'. User created ones are what matters?
        // Or specific count? 
        // Let's just show total count for simplicity, or total - system.
        // System folders usually have `system: true`.
        const userFolders = folders.filter(f => !f.system);
        folderCountEl.textContent = userFolders.length;
        linkCountEl.textContent = Object.keys(mappings || {}).length;
    }

    function showStatus(msg, type, persistent = false) {
        statusMsg.textContent = msg;
        statusMsg.style.color = type === 'error' ? '#d93025' : (type === 'success' ? '#188038' : '#5f6368');

        // Clear any existing timeout
        if (window.statusTimeout) clearTimeout(window.statusTimeout);

        if (!persistent) {
            window.statusTimeout = setTimeout(() => {
                statusMsg.textContent = '';
                // Restore sync status? Ideally yes, but simplified for now:
                // User action messages (export) will clear, but sync status should remain.
                // Re-calling loadStats() is overkill.
                // We'll just let it clear user actions.
                // BUT user wants Sync status FIXED.
                // If I clear "Exported", the status area becomes empty.
                // I should probably have TWO status areas? Or just revert to sync status.
                // For this edit, let's keep it simple: Persistent messages stay. 
                // Non-persistent ones clear to empty.
            }, 3000);
        }
    }

    // Export Logic
    btnExport.addEventListener('click', () => {
        chrome.storage.sync.get(null, (items) => {
            const dataStr = JSON.stringify(items, null, 2);
            const blob = new Blob([dataStr], { type: 'application/json' });
            const url = URL.createObjectURL(blob);

            const timestamp = new Date().toISOString().replace(/[:.]/g, '-').slice(0, 19);
            const filename = `gemini_links_backup_${timestamp}.json`;

            const a = document.createElement('a');
            a.href = url;
            a.download = filename;
            document.body.appendChild(a);
            a.click();
            document.body.removeChild(a);
            URL.revokeObjectURL(url);

            showStatus('已匯出備份', 'success');
        });
    });

    // Import Logic
    btnImport.addEventListener('click', () => {
        fileImport.click();
    });

    fileImport.addEventListener('change', (e) => {
        const file = e.target.files[0];
        if (!file) return;

        const reader = new FileReader();
        reader.onload = (event) => {
            try {
                const data = JSON.parse(event.target.result);
                if (typeof data !== 'object') throw new Error('Invalid JSON');

                chrome.storage.sync.set(data, () => {
                    showStatus('匯入成功！', 'success');
                    loadStats();
                    fileImport.value = '';
                });
            } catch (err) {
                console.error(err);
                showStatus('匯入失敗：格式錯誤', 'error');
            }
        };
        reader.readAsText(file);
    });
});
