// content.js - Gemini Link Organizer
// Security: All user inputs are sanitized to prevent XSS attacks
// Privacy: All data is stored locally in chrome.storage, no external server communication

const STORAGE_KEY_FOLDERS = 'folders';
const STORAGE_KEY_MAPPINGS = 'linkMappings';
const STORAGE_KEY_NOTES = 'linkNotes';

// Selectors for Gemini share links
const SELECTORS = {
    CARD: '.link-card, div[data-test-id="history-item"], a[href*="/share/"]',
    LINK: 'a[href*="/share/"]', // Matches g.co/gemini/share and gemini.google.com/share
    CARD_CONTAINER: '.public-links-list, .gds-card-list'
};

// ...

// State
let folders = [];
let linkMappings = {}; // { "url": "folderId" }
let linkNotes = {}; // { "url": "Note Content" }
let currentFilter = 'all'; // Default
let currentLinkSearch = ''; // New state for link title search

// Helper: Generate ID
function generateId() {
    return 'f_' + Math.random().toString(36).substr(2, 9);
}

// Helper: Sanitize text for safe display (defense in depth)
// This prevents potential XSS even if textContent is mistakenly changed to innerHTML
function sanitizeText(text) {
    if (typeof text !== 'string') return '';
    // Remove any potential HTML tags and trim
    return text.replace(/<[^>]*>/g, '').trim();
}

// Initialization
async function init() {
    await loadData();
    injectSidebar();
    observePage();
    processExistingCards();
}

// Data Loading
// Data Loading (Sync with Local Fallback/Migration)
async function loadData() {
    // 1. Try to get from Sync storage
    let data = await chrome.storage.sync.get([STORAGE_KEY_FOLDERS, STORAGE_KEY_MAPPINGS, STORAGE_KEY_NOTES]);

    // 2. Check if Sync is empty (first time run or new device)
    if (!data[STORAGE_KEY_FOLDERS] || data[STORAGE_KEY_FOLDERS].length === 0) {
        console.log('[GeminiOrganizer] Sync storage empty. Checking local storage for migration...');
        // 3. Check Local storage for existing data to migrate
        const localData = await chrome.storage.local.get([STORAGE_KEY_FOLDERS, STORAGE_KEY_MAPPINGS, STORAGE_KEY_NOTES]);

        if (localData[STORAGE_KEY_FOLDERS] && localData[STORAGE_KEY_FOLDERS].length > 0) {
            console.log('[GeminiOrganizer] Found local data. Migrating to Sync...');
            folders = localData[STORAGE_KEY_FOLDERS];
            linkMappings = localData[STORAGE_KEY_MAPPINGS] || {};
            linkNotes = localData[STORAGE_KEY_NOTES] || {};

            // Save to Sync immediately
            await saveData();
            // Optional: Clear local? better keep as backup for now or user manual clear.
        } else {
            // New user, initialize defaults
            folders = [
                { id: 'all', name: '全部顯示', system: true },
                { id: 'uncategorized', name: '未分類', system: true }
            ];
            linkMappings = {};
            linkNotes = {};
        }
    } else {
        folders = data[STORAGE_KEY_FOLDERS];
        linkMappings = data[STORAGE_KEY_MAPPINGS] || {};
        linkNotes = data[STORAGE_KEY_NOTES] || {};
    }

    // Double check system folders existence
    if (!folders.find(f => f.id === 'all')) folders.unshift({ id: 'all', name: '全部顯示', system: true });
    if (!folders.find(f => f.id === 'uncategorized')) folders.push({ id: 'uncategorized', name: '未分類', system: true });
}

async function saveData() {
    // Filter out system folders from storage to save space
    const foldersToSave = folders.filter(f => !f.system);

    const dataToSave = {
        [STORAGE_KEY_FOLDERS]: foldersToSave,
        [STORAGE_KEY_MAPPINGS]: linkMappings,
        [STORAGE_KEY_NOTES]: linkNotes
    };

    try {
        // Check storage quota before saving
        const dataSize = JSON.stringify(dataToSave).length;
        const QUOTA_BYTES_PER_ITEM = chrome.storage.sync.QUOTA_BYTES_PER_ITEM || 8192;

        if (dataSize > QUOTA_BYTES_PER_ITEM * 0.9) { // Warning at 90% capacity
            console.warn('[GeminiOrganizer] Storage nearing capacity:', dataSize, '/', QUOTA_BYTES_PER_ITEM);
            // Consider using local storage as fallback
            await chrome.storage.local.set(dataToSave);
            console.log('[GeminiOrganizer] Saved to local storage due to size constraints');
        } else {
            await chrome.storage.sync.set(dataToSave);
        }
    } catch (error) {
        console.error('[GeminiOrganizer] Failed to save data:', error);
        // Fallback to local storage on error
        try {
            await chrome.storage.local.set(dataToSave);
            console.log('[GeminiOrganizer] Fallback: saved to local storage');
        } catch (localError) {
            console.error('[GeminiOrganizer] Critical: Failed to save to local storage:', localError);
        }
    }
}

// Helper: Generate Pastel Color from String
function generatePastelColor(str) {
    let hash = 0;
    for (let i = 0; i < str.length; i++) {
        hash = str.charCodeAt(i) + ((hash << 5) - hash);
    }
    const h = Math.abs(hash) % 360;
    return `hsl(${h}, 70%, 93%)`; // High lightness for pastel
}

// UI: Sidebar
function injectSidebar() {
    if (document.getElementById('gemini-organizer-sidebar')) return;

    const sidebar = document.createElement('div');
    sidebar.id = 'gemini-organizer-sidebar';
    sidebar.innerHTML = `
        <div id="gemini-organizer-toggle" title="收合/展開">◀</div>
        <div class="sidebar-header">
            <h3>連結整理</h3>
        </div>
        
        <!-- Search Row (Side by Side) -->
        <div class="search-row">
            <input type="text" id="organizer-link-search" class="search-input" placeholder="🔍 搜尋連結" title="搜尋連結標題及備註">
            <input type="text" id="organizer-folder-search" class="search-input" placeholder="📂 搜尋分類" title="搜尋分類資料夾">
        </div>

        <!-- New Folder Section -->
        <div class="new-folder-section">
            <div class="new-folder-form">
                <input type="text" id="organizer-new-folder-name" class="new-folder-input" placeholder="新增資料夾...">
                <button id="organizer-add-folder-btn" class="add-folder-btn">+</button>
            </div>
        </div>

        <div class="sidebar-content" id="organizer-folder-list">
            <!-- Folders will go here -->
        </div>

        <div class="sidebar-footer">
            Made by <a href="https://kentxchang.blogspot.tw" target="_blank">阿剛老師</a>
            <br>
            <span class="cc-license">CC BY-NC-SA 4.0</span>
        </div>
    `;

    document.body.appendChild(sidebar);

    // Events
    document.getElementById('organizer-add-folder-btn').addEventListener('click', () => {
        const input = document.getElementById('organizer-new-folder-name');
        const name = sanitizeText(input.value); // Sanitize user input
        if (name) {
            addFolder(name);
            input.value = '';
        }
    });

    // Folder Search Event
    document.getElementById('organizer-folder-search').addEventListener('input', (e) => {
        renderSidebarGroups(e.target.value.trim()); // Pass search term
    });

    // Link Search Event (New)
    document.getElementById('organizer-link-search').addEventListener('input', (e) => {
        currentLinkSearch = e.target.value.trim().toLowerCase();
        refreshVisibility();
    });

    // Toggle collapse
    const toggleBtn = document.getElementById('gemini-organizer-toggle');
    toggleBtn.addEventListener('click', (e) => {
        e.stopPropagation();
        sidebar.classList.toggle('collapsed');
        // Icon logic might need reversing depending on visual preference, but standard arrow flip:
        toggleBtn.textContent = sidebar.classList.contains('collapsed') ? '▶' : '◀';
    });

    // Allow enter key
    document.getElementById('organizer-new-folder-name').addEventListener('keypress', (e) => {
        if (e.key === 'Enter') document.getElementById('organizer-add-folder-btn').click();
    });

    renderSidebarGroups();
}

function addFolder(name) {
    const newFolder = { id: generateId(), name: name };
    folders.push(newFolder);
    saveData();
    renderSidebarGroups();
    // Update all dropdowns on the page
    updateAllDropdowns();
}

function deleteFolder(id) {
    if (!confirm('確定要刪除此資料夾嗎？')) return;

    folders = folders.filter(f => f.id !== id);
    // Remove mappings for this folder
    Object.keys(linkMappings).forEach(url => {
        if (linkMappings[url] === id) {
            delete linkMappings[url];
        }
    });

    saveData();
    renderSidebarGroups();
    updateAllDropdowns();
    applyFilter('all'); // Reset filter if current deleted
}

function renderSidebarGroups(searchTerm = '') {
    const container = document.getElementById('organizer-folder-list');
    container.innerHTML = '';

    const lowerTerm = searchTerm.toLowerCase();

    folders.forEach(folder => {
        // Filter logic
        if (lowerTerm && !folder.name.toLowerCase().includes(lowerTerm)) {
            return; // Skip if doesn't match
        }

        const div = document.createElement('div');
        div.className = `folder-item ${currentFilter === folder.id ? 'active' : ''}`;
        div.dataset.id = folder.id;
        div.title = folder.name; // Tooltip for full name

        // Apply pastel color
        if (currentFilter !== folder.id) {
            div.style.backgroundColor = generatePastelColor(folder.name);
        } else {
            // Active state already handles color relative to theme or maybe we darken it?
            // Let's stick to CSS for active, and inline for inactive.
        }

        // Create folder name span with textContent to prevent XSS
        const nameSpan = document.createElement('span');
        nameSpan.textContent = folder.name; // Safe: uses textContent instead of innerHTML
        div.appendChild(nameSpan);

        // Add delete button for non-system folders
        if (!folder.system) {
            const actionsDiv = document.createElement('div');
            actionsDiv.className = 'folder-actions';

            const deleteBtn = document.createElement('button');
            deleteBtn.className = 'folder-action-btn delete-btn';
            deleteBtn.dataset.id = folder.id;
            deleteBtn.title = '刪除';
            deleteBtn.textContent = '🗑️';

            actionsDiv.appendChild(deleteBtn);
            div.appendChild(actionsDiv);
        }

        div.addEventListener('click', (e) => {
            // Avoid triggering if clicking delete
            if (e.target.closest('.delete-btn')) return;
            applyFilter(folder.id);
        });

        // Delete handler
        const delBtn = div.querySelector('.delete-btn');
        if (delBtn) {
            delBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                deleteFolder(folder.id);
            });
        }

        container.appendChild(div);
    });
}

// UI: Filtering / Visibility
function applyFilter(folderId) {
    currentFilter = folderId;
    renderSidebarGroups(); // Update active class
    refreshVisibility();
}

function refreshVisibility() {
    // Find all links again to toggle visibility
    const links = document.querySelectorAll(SELECTORS.LINK);
    links.forEach(link => {
        let card = link.closest('.link-card') || link.closest('gds-card') || link.closest('li') || link.closest('[role="listitem"]');
        if (!card) card = link.parentElement;
        if (card === document.body || card === link) {
            card = link.parentElement;
        }

        const url = link.href;
        const mappedFolder = linkMappings[url];

        // 1. Check Folder Filter
        let folderMatch = true;
        if (currentFilter === 'all' || !currentFilter) { // Handle null or 'all'
            folderMatch = true;
        } else if (currentFilter === 'uncategorized') {
            folderMatch = !mappedFolder;
        } else {
            folderMatch = (mappedFolder === currentFilter);
        }

        // 2. Check Link Title Search
        let titleMatch = true;
        if (currentLinkSearch) {
            // Find text content in the card
            // Note: Since we inject the note into the card DOM (e.g. in a span), 
            // innerText should automatically catch it!
            const text = card ? card.innerText.toLowerCase() : '';
            if (!text.includes(currentLinkSearch)) {
                titleMatch = false;
            }
        }

        const visible = folderMatch && titleMatch;

        // Hide the container
        if (card) card.style.display = visible ? '' : 'none';
    });
}

// DOM Observation
let observerTimeout;
function observePage() {
    const observer = new MutationObserver((mutations) => {
        // Filter out mutations that are our own doing to avoid loops
        const shouldProcess = mutations.some(mutation => {
            const target = mutation.target;
            // Ignore changes to our sidebar
            if (target.id === 'gemini-organizer-sidebar' || target.closest('#gemini-organizer-sidebar')) return false;
            // Ignore changes to our injected dropdowns
            if (target.classList && target.classList.contains('organizer-selector-container')) return false;
            // Ignore changes inside our dropdowns
            if (target.closest('.organizer-selector-container')) return false;

            // For added nodes, check if they are ours
            if (mutation.addedNodes.length > 0) {
                for (let i = 0; i < mutation.addedNodes.length; i++) {
                    const node = mutation.addedNodes[i];
                    if (node.nodeType !== 1) continue; // Ignore text nodes often
                    if (node.id === 'gemini-organizer-sidebar') return false;
                    if (node.className && typeof node.className === 'string' && node.className.includes('organizer-selector-container')) return false;
                }
                return true; // Genuine new content
            }
            return false;
        });

        if (shouldProcess) {
            clearTimeout(observerTimeout);
            // Debounce to prevent rapid firing during layout trashing
            observerTimeout = setTimeout(() => {
                processExistingCards();
            }, 500);
        }
    });

    observer.observe(document.body, { childList: true, subtree: true });
}

function processExistingCards() {
    console.log('[GeminiOrganizer] Scanning for links...');
    const links = document.querySelectorAll(SELECTORS.LINK);
    console.log(`[GeminiOrganizer] Found ${links.length} potential links.`);

    links.forEach(link => {
        // Skip links that are already inside our own sidebar or tools
        if (link.closest('#gemini-organizer-sidebar')) return;

        // Check if THIS specific link has been processed
        // We might need to re-process if we update notes logic? 
        // For now, let's assume we can re-inject purely UI parts if needed, 
        // but easier to check dataset.
        if (link.dataset.geminiOrganizerProcessed === 'true') {
            // Check if Note UI is missing (e.g. old version of script processed it)
            // If missing, we continue to injection logic, but need to be careful not to duplicate dropdowns.
            // Let's rely on injectCardControls being smart or checking presence.

            // Find container next to link?
            let card = link.closest('.link-card') || link.closest('gds-card');
            if (!card) {
                card = link.closest('li') || link.closest('[role="listitem"]');
                if (!card) card = link.parentElement;
                if (card === document.body) card = link;
            }

            const existingBtn = card.querySelector('.organizer-note-btn');
            if (!existingBtn) {
                console.log('[GeminiOrganizer] Reprocessing link to add missing UI:', link.href);
                // Continue to injection...
            } else {
                return; // Fully processed
            }
        }

        // Try to find a nice card container for styling context, 
        // but if we fail or if it's shared, we will just focus on the link.
        let card = link.closest('.link-card') || link.closest('gds-card');

        if (!card) {
            // Heuristic: Go up until we hit a block element that acts as a row.
            // But stop before we hit a container with multiple such links if possible.
            // Safest bet for generic injection is just the immediate parent or grandparent
            // but specific enough to be "the row".

            // Try to find an LI or role=listitem
            card = link.closest('li') || link.closest('[role="listitem"]');

            // If not, use parentElement (usually the wrapper of the text)
            if (!card) card = link.parentElement;

            // If parent is body or main, that's too far. 
            if (card === document.body) card = link;
        }

        console.log('[GeminiOrganizer] Processing link:', link.href);
        injectCardControls(card, link, link.href);

        // Mark link as processed so we don't duplicate
        link.dataset.geminiOrganizerProcessed = 'true';
    });

    if (currentFilter) {
        applyFilter(currentFilter);
    }
}

function injectCardControls(card, linkElement, url) {
    // Determine injection target (Header or Link)
    const header = card.querySelector('h3, h4, h2, div[role="heading"]');

    // Attempt to find existing container
    // If we injected it before, it should be next sibling of header, or next sibling of linkElement.
    // Let's look for it in the card specifically.
    // Be careful not to grab a container for a DIFFERENT link if multiple links are in same card (unlikely with this logic, but possible).
    // Safest: exact same traversal logic.

    let existingContainer = null;
    if (header) {
        if (header.nextElementSibling && header.nextElementSibling.classList.contains('organizer-selector-container')) {
            existingContainer = header.nextElementSibling;
        }
    } else {
        if (linkElement.nextElementSibling && linkElement.nextElementSibling.classList.contains('organizer-selector-container')) {
            existingContainer = linkElement.nextElementSibling;
        }
    }

    if (existingContainer) {
        // Container exists. Check for button.
        if (!existingContainer.querySelector('.organizer-note-btn')) {
            // Append Button
            const noteBtn = document.createElement('button');
            noteBtn.className = 'organizer-note-btn';
            noteBtn.title = '編輯備註';
            noteBtn.textContent = '✎';
            noteBtn.addEventListener('click', (e) => handleNoteClick(e, url, card));
            existingContainer.appendChild(noteBtn);
        }
        // Ensure display is updated
        updateLinkDisplay(card, url);
        return;
    }

    // Create New Container
    const container = document.createElement('div');
    container.className = 'organizer-selector-container';
    container.style.marginTop = '8px';
    container.style.marginBottom = '8px';

    container.innerHTML = `
        <select class="organizer-folder-select" title="設定資料夾">
            <option value="">(無分類)</option>
        </select>
        <button class="organizer-note-btn" title="編輯備註">✎</button>
    `;

    const select = container.querySelector('select');
    populateDropdown(select, url);

    select.addEventListener('change', (e) => {
        e.stopPropagation(); // Prevent card click
        const newVal = e.target.value;
        if (newVal) {
            linkMappings[url] = newVal;
        } else {
            delete linkMappings[url];
        }
        saveData();
        if (currentFilter && currentFilter !== 'all') {
            applyFilter(currentFilter);
        }
    });

    select.addEventListener('click', (e) => e.stopPropagation());

    // Note Button Listener
    const noteBtn = container.querySelector('.organizer-note-btn');
    noteBtn.addEventListener('click', (e) => handleNoteClick(e, url, card));

    // Injection
    if (header) {
        // Insert after header
        header.parentElement.insertBefore(container, header.nextSibling);
    } else {
        // Insert after link
        linkElement.insertAdjacentElement('afterend', container);
    }

    // Initial Display of Note
    updateLinkDisplay(card, url);
}

function handleNoteClick(e, url, card) {
    e.stopPropagation();
    e.preventDefault();

    const currentNote = linkNotes[url] || '';
    const newNote = prompt('編輯備註 (顯示於標題後方)：', currentNote);

    if (newNote !== null) { // If not cancelled
        const sanitized = sanitizeText(newNote); // Sanitize user input
        if (sanitized) {
            linkNotes[url] = sanitized;
        } else {
            delete linkNotes[url];
        }
        saveData();
        updateLinkDisplay(card, url);
    }
}

// Helper to update the link title with the note in ()
function updateLinkDisplay(card, url) {
    // Find the title element - prioritize headers and avoid link elements
    let target = card.querySelector('h3, h4, h2, h1, div[role="heading"]');

    // If no standard header, try to find title-like elements
    if (!target) {
        target = card.querySelector('.title, .card-title, [class*="title"]');
    }

    // Do NOT use the link element as target - we only want to show notes in titles
    // If there's no title element found, skip adding the note display
    if (!target || target.tagName === 'A') return;

    // Try to find the timestamp/creation time element
    let timeElement = null;
    const possibleTimeElements = card.querySelectorAll('div, span, p, time');
    for (let elem of possibleTimeElements) {
        const text = elem.textContent || '';
        if (text.includes('建立時間') || text.match(/\d{4}年.*月.*日/)) {
            if (elem.children.length <= 1) {
                timeElement = elem;
                break;
            }
        }
    }

    // Remove existing note field if any (search in the entire card to handle edge cases)
    const existingNoteField = card.querySelector('.gemini-organizer-note-field');
    if (existingNoteField) existingNoteField.remove();

    const note = linkNotes[url];
    if (note) {
        // Create the note field
        const noteField = document.createElement('span');
        noteField.className = 'gemini-organizer-note-field';
        noteField.textContent = ` - ${note}`;
        noteField.style.color = '#5f6368';
        noteField.style.fontWeight = 'normal';
        noteField.style.fontSize = '0.9em';

        noteField.style.marginLeft = '8px';
        noteField.style.display = 'inline';

        // Insert after time element if found, otherwise fallback to title
        if (timeElement) {
            timeElement.insertAdjacentElement('afterend', noteField);
        } else {
            // Fallback: find title and insert after it
            let target = card.querySelector('h3, h4, h2, h1, div[role="heading"]');
            if (target && target.tagName !== 'A') {
                target.insertAdjacentElement('afterend', noteField);
            }
        }
    }
}

function populateDropdown(select, url) {
    // Save current selection
    // Clear options except default
    while (select.options.length > 1) {
        select.remove(1);
    }

    const currentFolderId = linkMappings[url];

    folders.forEach(folder => {
        if (folder.system) return; // Don't show "All" in dropdown
        const option = document.createElement('option');
        option.value = folder.id;
        option.textContent = folder.name;
        if (currentFolderId === folder.id) {
            option.selected = true;
        }
        select.appendChild(option);
    });
}

function updateAllDropdowns() {
    // Only update cards that have been processed (have selectors)
    const cards = document.querySelectorAll('.link-card, .gds-card, [role="listitem"]');
    cards.forEach(card => {
        const link = card.querySelector(SELECTORS.LINK);
        if (!link) return;

        // Skip if not yet processed
        if (link.dataset.geminiOrganizerProcessed !== 'true') return;

        const select = card.querySelector('.organizer-folder-select');
        if (select) populateDropdown(select, link.href);

        // Also update note display
        updateLinkDisplay(card, link.href);
    });
}

// Listen for storage changes (e.g. from Sync or Import in Popup)
chrome.storage.onChanged.addListener((changes, namespace) => {
    if (namespace === 'sync' || namespace === 'local') {
        if (changes[STORAGE_KEY_FOLDERS] || changes[STORAGE_KEY_MAPPINGS] || changes[STORAGE_KEY_NOTES]) {
            console.log('[GeminiOrganizer] Storage changed, reloading data...');
            loadData().then(() => {
                renderSidebarGroups();
                updateAllDropdowns();
                refreshVisibility();
            });
        }
    }
});

// Start
init();
